<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DTOs\Brands\BrandCreateDto;
use App\DTOs\Brands\BrandUpdateDto;
use App\Repositories\Interfaces\IBrandRepository;
use Exception;
use App\Models\Brand;

class BrandController extends Controller
{

    private $brandRepository;

    private $errorConnection = "";

    public function __construct(Request $request, IBrandRepository $brandRepository)
    {
        if (isset($request['business_information'])) {
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $existe_conexion = $this->setDBClient($ruc);

            if($existe_conexion['error']){
                $this->errorConnection=$existe_conexion['message'];
                return $this->messageError("Se espera un número de RUC válido.", 201); 
            } else {
                $this->brandRepository = $brandRepository;
            }
        } else {
            $this->errorConnection='Error en la conexión. Se espera número de RUC';
        }     
    }

    public function index(Request $request)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201); 
        }

        $result = $this->brandRepository->paginate($request->all());

        if ($result) {
            return $this->successResponse($result, "Marcas encontrados correctamente", 201);
        } else {
            return $this->errorResponse("No hay Marcas", 201);
        }
    }

    public function show(int $id)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201); 
        }

        $result = $this->brandRepository->find($id);

        if ($result) {
            return $this->successResponse($result, "Marca encontrada correctamente", 201);
        } else {
            return $this->errorResponse("Marca no encontrada", 201);
        }
    }

    public function store(Request $request)
    {
        try {
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201); 
            }
    
            $code = $request['code'];
            $codeExist = Brand::where('code', '=', $code)->get();

            if (count($codeExist) == 0) {
                $this->validate($request, [
                    'code' => 'required|max:10',
                    'brand' => 'required|max:250'
                ], [
                    'code.required' => 'El código de Marca es requerido',
                    'code.max' => 'El código de Marca no puede superar los :max caracteres',
                    'brand.required' => 'El nombre de Marca es requerido'
                ]);

                $store = new BrandCreateDto($request->except('business_information'));
                $result = $this->brandRepository->store($store);
        
                return $this->successResponse($result, "Marca creada correctamente", 201);
            } else {
                return $this->errorResponse("Codigo de Marca ingresado ya existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 201);
        }        
    }

    public function update(int $id, Request $request)
    {
        try {
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201); 
            }
    
            $result = $this->brandRepository->find($id);

            if ($result) {
                $code = $request['code'];
                $codeExist = Brand::where('id', '!=', $id)->where('code', '=', $code)->get();
    
                if (count($codeExist) == 0) {
                    $this->validate($request, [
                        'code' => [
                            'required',
                            'max:10'
                        ],
                        'brand' => 'required|max:250'
                    ], [
                        'code.required' => 'El código de Marca es requerido',
                        'code.max' => 'El código de Marca no puede superar los :max caracteres',
                        'brand.required' => 'El nombre de Marca es requerido'
                    ]);
            
                    $data = $request->except('business_information');
                    $data['id'] = $id;
            
                    $entry = new BrandUpdateDto($data);
            
                    $this->brandRepository->update($entry);
    
                    $result = $this->brandRepository->find($id);
    
                    return $this->successResponse($result, "Marca actualizada correctamente", 201);
                } else {
                    return $this->errorResponse("Codigo de Marca ingresado ya existe", 201);
                }
            } else {
                return $this->errorResponse("Marca ingresada no existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 201);
        }
    }

    public function destroy(int $id)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201); 
        }

        $result = $this->brandRepository->find($id);

        if ($result) {
            $this->brandRepository->destroy($id);

            return $this->successResponse($id, "Marca eliminada correctamente", 201);    
        } else {
            return $this->errorResponse("Marca ingresada no existe", 201);
        }
    }    
}
