<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DTOs\TypesMovement\TypeMovementCreateDto;
use App\DTOs\TypesMovement\TypeMovementUpdateDto;
use App\Repositories\Interfaces\ITypeMovementRepository;
use Exception;
use App\Models\TypeMovement;

class TypeMovementController extends Controller
{

    private $typemovementRepository;

    private $errorConnection = "";

    public function __construct(Request $request, ITypeMovementRepository $typemovementRepository)
    {
        if (isset($request['business_information'])) {
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $existe_conexion = $this->setDBClient($ruc);

            if($existe_conexion['error']){
                $this->errorConnection=$existe_conexion['message'];
                return $this->messageError("Se espera un número de RUC válido.", 201); 
            } else {
                $this->typemovementRepository = $typemovementRepository;
            }
        } else {
            $this->errorConnection='Error en la conexión. Se espera número de RUC';
        }   
    }

    public function index(Request $request)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201); 
        }

        $result = $this->typemovementRepository->paginate($request->all());

        if ($result) {
            return $this->successResponse($result, "Tipos de Movimiento encontrados correctamente", 201);
        } else {
            return $this->errorResponse("No hay Tipos de Movimiento", 201);
        }
    }

    public function show(int $id)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201); 
        }

        $result = $this->typemovementRepository->find($id);

        if ($result) {
            return $this->successResponse($result, "Tipo de Movimiento encontrado correctamente", 201);
        } else {
            return $this->errorResponse("Tipo de Movimiento no encontrado", 201);
        }
    }

    public function store(Request $request)
    {
        try {
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201); 
            }

            $code = $request['code'];
            $codeExist = TypeMovement::where('code', '=', $code)->get();

            if (count($codeExist) == 0) {
                $this->validate($request, [
                    'code' => 'required|max:10',
                    'motive' => 'required|max:250',
                    'alias' => 'required|max:10',
                    'type' => 'required|max:10',
                    'purchase' => 'required|max:1',
                    'billable' => 'required|max:1',
                    'code_sunat' => 'required|max:8'
                ], [
                    'code.required' => 'El código de Tipo de Movimiento es requerido',
                    'code.max' => 'El código de Tipo de Movimiento no puede superar los :max caracteres',
                    'motive.required' => 'El nombre de Tipo de Movimiento es requerido',
                    'alias.required' => 'El alias de Tipo de Movimiento es requerido',
                    'type.required' => 'El tipo de Tipo de Movimiento es requerido',
                    'purchase.required' => 'Es valor Compra de Tipo de Movimiento es requerido',
                    'billable.required' => 'Es valor de Facturable de Tipo de Movimiento es requerido',
                    'code_sunat.required' => 'Es Codigo de SUNAT del Tipo de Movimiento es requerido'
                ]);
    
                $store = new TypeMovementCreateDto($request->except('business_information'));
                $result = $this->typemovementRepository->store($store);
        
                return $this->successResponse($result, "Tipo de Movimiento creado correctamente", 201);    
            } else {
                return $this->errorResponse("Codigo de Almacen ingresado ya existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 201);
        }        
    }

    public function update(int $id, Request $request)
    {
        try {
            if($this->errorConnection != ''){
                return $this->messageError($this->errorConnection, 201); 
            }

            $result = $this->typemovementRepository->find($id);

            if ($result) {
                $code = $request['code'];
                $codeExist = TypeMovement::where('id', '!=', $id)->where('code', '=', $code)->get();
    
                if (count($codeExist) == 0) {
                    $this->validate($request, [
                        'code' => [
                            'required',
                            'max:10'
                        ],
                        'motive' => 'required|max:250',
                        'alias' => 'required|max:10',
                        'type' => 'required|max:10',
                        'purchase' => 'required|max:1',
                        'billable' => 'required|max:1',
                        'code_sunat' => 'required|max:8'
                    ], [
                        'code.required' => 'El código de Tipo de Movimiento es requerido',
                        'code.max' => 'El código de Tipo de Movimiento no puede superar los :max caracteres',
                        'motive.required' => 'El nombre de Tipo de Movimiento es requerido',
                        'alias.required' => 'El alias de Tipo de Movimiento es requerido',
                        'type.required' => 'El tipo de Tipo de Movimiento es requerido',
                        'purchase.required' => 'Es valor Compra de Tipo de Movimiento es requerido',
                        'billable.required' => 'Es valor de Facturable de Tipo de Movimiento es requerido',
                        'code_sunat.required' => 'Es Codigo de SUNAT del Tipo de Movimiento es requerido'
                    ]);
            
                    $data = $request->except('business_information');
                    $data['id'] = $id;
            
                    $entry = new TypeMovementUpdateDto($data);
            
                    $this->typemovementRepository->update($entry);
        
                    $result = $this->typemovementRepository->find($id);
        
                    return $this->successResponse($result, "Tipo de Movimiento actualizado correctamente", 201);  
                } else {
                    return $this->errorResponse("Codigo de Almacen ingresado ya existe", 201);
                }
            } else {
                return $this->errorResponse("Tipo de Movimiento ingresado no existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 201);
        }
    }

    public function destroy(int $id)
    {
        if($this->errorConnection != ''){
            return $this->messageError($this->errorConnection, 201); 
        }

        $result = $this->typemovementRepository->find($id);

        if ($result) {
            $this->typemovementRepository->destroy($id);

            return $this->successResponse($id, "Tipo de Movimiento eliminado correctamente", 201);    
        } else {
            return $this->errorResponse("Tipo de Movimiento ingresado no existe", 201);
        }
    }    
}
