<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\DTOs\Trader\TraderCreateDto;
use App\DTOs\Trader\TraderUpdateDto;
use Exception;
use App\Repositories\Interfaces\ITraderRepository;
use App\Models\Trader;

class TraderController extends Controller
{

    private $traderRepository;
    private $errorConnection = "";

    public function __construct(Request $request, ITraderRepository $traderRepository)
    {
        if (isset($request['business_information'])) {
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $existe_conexion = $this->setDBClient($ruc);

            if ($existe_conexion['error']) {
                $this->errorConnection = $existe_conexion['message'];
                return $this->messageError("Se espera un numero de RUC valido.", 422);
            } else {
                $this->traderRepository = $traderRepository;
            }
        } else {
            $this->errorConnection = 'Error en la conexion. Se espera numero de RUC';
        }
    }

    public function index(Request $request)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->traderRepository->paginate($request->all());

        if ($result) {
            if ($result['total'] > 0) {
                return $this->successResponse($result, "Vendedores encontrados correctamente.", 200);
            } else {
                return $this->successResponse($result, "No hay Vendedores cargadas en el sistema.", 200);
            }
            //return $this->successResponse($result, "Marcas encontrados correctamente", 201);
        } else {
            return $this->errorResponse("No hay Vendedores.", 404);
        }
    }

    public function show(int $id)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->traderRepository->find($id);

        if ($result) {
            return $this->successResponse($result, "Vendedor encontrada correctamente", 200);
        } else {
            return $this->errorResponse("Vendedor no encontrada", 200);
        }
    }

    public function store(Request $request)
    {
        try {
            if ($this->errorConnection != '') {
                return $this->messageError($this->errorConnection, 500);
            }
            $business_information = $request['business_information'];
            $ruc = $business_information['ruc_number'];

            $code = $request['cod_vend'];
            $type_document = $request['type_document'];
            $document = $request['document'];
            //VALIDACION 1
            $codeAndRucExist = Trader::where('cod_vend', '=', $code)->where('ruc', '=', $ruc)->where('status', '=', 1)->get();
            //VALIDACION 2
            $rucAndTydocumentAndDocument = Trader::where('ruc', '=', $ruc)->where('type_document', '=', $type_document)->where('document', '=', $document)->where('status', '=', 1)->get();
            //Validación de ruc, tipo de documento y documento.
            if (count($rucAndTydocumentAndDocument) >= 1) {
                return $this->errorResponse("El tipo de documento y documento ingresado ya existe", 422);
            }
            if (count($codeAndRucExist) >= 1) {
                return $this->errorResponse("Codigo de vendedor ingresado ya existe", 422);
            }
            $this->validate($request, [
                //'ruc' => 'required|numeric|max:11',
                'cod_vend' => 'required|alpha_dash|max:10',
                'type_document' => 'required|numeric|digits_between:1,11',
                'document' => 'required|alpha_dash|max:15',
                'name' => 'required|max:50',
                'last_name_1' => 'required|max:30',
                'birth_date' => 'required|date',
                'gender' => 'required|max:1|in:M,F,O',
                'ubigeo' => 'numeric|digits_between:1,11',
                'movil' => 'numeric|digits_between:1,20',
                'movil_2' => 'numeric|digits_between:1,20',
                'email' => 'email',
                'user_id' => 'numeric|digits_between:1,11',
                'type_commission' => 'in:PV,PG,MF',
                'valor_commission' => 'regex:/^[0-9]{1,10}(,[0-9]{1})*(\.[0-9]{1,4}+)*$/',
                'created_by' => 'required|numeric|digits_between:1,11',
            ], [
                'cod_vend.alpha_dash' => 'El codigo de Vendedor no puede tener caracteres especiales.',
                'cod_vend.required' => 'El codigo de Vendedor es requerido.',
                'cod_vend.max' => 'El codigo de Vendedor no puede superar los :max caracteres.',
                'type_document.required' => 'El tipo de documento es requerido.',
                'type_document.max' => 'El tipo de documento no puede superar los :max caracteres.',
                'type_document.numeric' => 'El tipo de documento debe ser numerico',
                'document.required' => 'El documento es requerido.',
                'document.max' => 'El documento no puede superar los :max caracteres.',
                'document.alpha_dash' => 'El documento no puede tener caracteres especiales.',
                'name.required' => 'El nombre es requerido.',
                'name.max' => 'El nombre no puede superar los :max caracteres.',
                'birth_date.required' => 'La fecha de nacimiento es requerido.',
                'birth_date.date' => 'La fecha de nacimiento debe tener el formato de fecha.',
                'gender.in' => 'El genero es invalido.',
                'ubigeo.numeric' => 'El codigo ubigeo debe ser numerico.',
                'email.email' => 'El email debe ser valido.',
                'type_commission.in' => 'El tipo de comision es invalido.',
                'valor_commission.regex' => 'El formato del valor de comision es invalido.',
                'movil.numeric' => 'el movil debe ser numerico',
                'movil_2.numeric' => 'El movil 2 debe ser numerico',
                'user_id.numeric' => 'El campo user_id debe ser numerico.',
                'created_by.required' => 'El campo created_by es requerido.',
                'created_by.numeric' => 'El campo created_by debe ser numerico.',
                'created_by.digits_between' => 'El campo created_by no puede superar los :max caracteres.',
            ]);
            $store = new TraderCreateDto($request->except(array('business_information', 'items')));
            $store->ruc = $ruc;
            $result = $this->traderRepository->store($store);
            return $this->successResponse($result, "El vendedor fue creada correctamente", 201);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }

    public function update(int $id, Request $request)
    {
        try {

            if ($this->errorConnection != '') {
                return $this->messageError($this->errorConnection, 500);
            }


            $result = $this->traderRepository->find($id);

            if ($result) {
                $business_information = $request['business_information'];
                $ruc = $business_information['ruc_number'];

                $code = $request['cod_vend'];
                $type_document = $request['type_document'];
                $document = $request['document'];

                //VALIDACION 1
                $codeAndRucExist = Trader::where('cod_vend', '=', $code)->where('ruc', '=', $ruc)->where('id', '!=', $id)->where('status', '=', 1)->get();
                //VALIDACION 2
                $rucAndTydocumentAndDocument = Trader::where('ruc', '=', $ruc)->where('type_document', '=', $type_document)->where('document', '=', $document)->where('id', '!=', $id)->where('status', '=', 1)->get();
                //Validación de ruc, tipo de documento y documento.
                if (count($rucAndTydocumentAndDocument) >= 1) {
                    return $this->errorResponse("El ruc, tipo de documento y documento ingresado ya existe", 422);
                }
                if (count($codeAndRucExist) >= 1) {
                    return $this->errorResponse("Codigo de vendedor y ruc ingresado ya existe", 422);
                }
                $this->validate($request, [
                    //'ruc' => 'required|numeric|max:11',
                    'cod_vend' => 'required|alpha_dash|max:10',
                    'type_document' => 'required|numeric|digits_between:1,11',
                    'document' => 'required|alpha_dash|max:15',
                    'name' => 'required|max:50',
                    'last_name_1' => 'required|max:30',
                    'birth_date' => 'required|date',
                    'gender' => 'required|max:1|in:M,F,O',
                    'ubigeo' => 'numeric|digits_between:1,11',
                    'movil' => 'numeric|digits_between:1,20',
                    'movil_2' => 'numeric|digits_between:1,20',
                    'email' => 'email',
                    'user_id' => 'numeric|digits_between:1,11',
                    'type_commission' => 'in:PV,PG,MF',
                    'valor_commission' => 'regex:/^[0-9]{1,10}(,[0-9]{1})*(\.[0-9]{1,4}+)*$/',
                    'updated_by' => 'required|numeric|digits_between:1,11',
                ], [
                    'cod_vend.alpha_dash' => 'El codigo de Vendedor no puede tener caracteres especiales.',
                    'cod_vend.required' => 'El codigo de Vendedor es requerido.',
                    'cod_vend.max' => 'El codigo de Vendedor no puede superar los :max caracteres.',
                    'type_document.required' => 'El tipo de documento es requerido.',
                    'type_document.max' => 'El tipo de documento no puede superar los :max caracteres.',
                    'type_document.numeric' => 'El tipo de documento debe ser numerico',
                    'document.required' => 'El documento es requerido.',
                    'document.max' => 'El documento no puede superar los :max caracteres.',
                    'document.alpha_dash' => 'El documento no puede tener caracteres especiales.',
                    'name.required' => 'El nombre es requerido.',
                    'name.max' => 'El nombre no puede superar los :max caracteres.',
                    'birth_date.required' => 'La fecha de nacimiento es requerido.',
                    'birth_date.date' => 'La fecha de nacimiento debe tener el formato de fecha.',
                    'gender.in' => 'El genero es invalido.',
                    'ubigeo.numeric' => 'El codigo ubigeo debe ser numerico.',
                    'email.email' => 'El email debe ser valido.',
                    'type_commission.in' => 'El tipo de comision es invalido.',
                    'valor_commission.regex' => 'El formato del valor de comision es invalido.',
                    'movil.numeric' => 'el movil debe ser numerico',
                    'movil_2.numeric' => 'El movil 2 debe ser numerico',
                    'user_id.numeric' => 'El campo user_id debe ser numerico.',
                    'updated_by.required' => 'El campo updated_by es requerido.',
                    'updated_by.numeric' => 'El campo updated_by debe ser numerico.',
                    'updated_by.digits_between' => 'El campo updated_by no puede superar los :max caracteres.',
                ]);

                $data = $request->except('business_information');
                $data['id'] = $id;

                $entry = new TraderUpdateDto($data);

                $this->traderRepository->update($entry);

                $result = $this->traderRepository->find($id);

                return $this->successResponse($result, "El vendedor fue actualizado correctamente", 201);
            } else {
                return $this->errorResponse("Vendedor ingresado no existe", 201);
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->errorResponse($e->response->original, 422);
        }
    }

    public function destroy(int $id)
    {
        if ($this->errorConnection != '') {
            return $this->messageError($this->errorConnection, 500);
        }

        $result = $this->traderRepository->find($id);

        if ($result) {
            $this->traderRepository->destroy($id);

            return $this->successResponse($id, "Vendedor eliminado correctamente", 200);
        } else {
            return $this->errorResponse("Vendedor ingresado no existe", 200);
        }
    }
}
